function [tout, yout, P1, P2] = FEM_Solver_Parabolic( dt, A, k, C11, C12, C21, C22, cinitial, tfinal, tstep)
%Run a FEM approximation for diffusion. Space bdy nodes constant values.
%   Parabolic problem is of type Au_t - div[kC*grad u] = 0. UNLIKE other
%   codes, here the A - C22 are column vectors which correspond to the
%   nodal values of the equation coefficients ordered like Nodes under
%   Lists. The same is for cinitial which gives the nodal sample of the
%   initial distribution.  cbd indicates the values of nodes along the
%   boundary.  We simiplify the code and take the boundary values as fixed
%   in time.  The diffusion starts at time t = 0 and runs until tfinal
%   along tstep.  tout is a column vector of time values. yout(:,j) are
%   nodal values at time t = tout(j,1); 

%NOTE: This program was written sufficiently general that the k value may
%actually be a matrix.  For diffusion lab purposes.  


%Generate lists
tri = dt.Triangulation;
vert = dt.X;
[Edge, Etri, Boundary, Vmem, VNodal, ENodal, Nodal, Nodal_Bud, Nodes, Ntri, Nmem] = Lists(tri, vert);

%Rewrite the diffusion coefficient as k times identity matrix.
B11 = k;
B12 = k - k;
B21 = k-k;
B22 = k;

%From our initial lists extract the boundary data from the interior data.
cbd = cinitial(Nodal_Bud(2,1)+1:Nodal_Bud(4,1),1);
cinitial = cinitial(1:Nodal_Bud(2,1),:);

%Reformat the A-C22 terms as expected by Stiff Prin
A = Q_FNodal(Ntri,Nodes,A);
B11 = Q_FNodal(Ntri,Nodes,B11);
B12 = Q_FNodal(Ntri,Nodes,B12);
B21 = Q_FNodal(Ntri,Nodes,B21);
B22 = Q_FNodal(Ntri,Nodes,B22);
C11 = Q_FNodal(Ntri,Nodes,C11);
C12 = Q_FNodal(Ntri,Nodes,C12);
C21 = Q_FNodal(Ntri,Nodes,C21);
C22 = Q_FNodal(Ntri,Nodes,C22);



%Seperate Variables and left with P1c' + P2c = 0. Extract the matrices P1
%and P2.  

[P1, P2] = Stiff_Prin(dt,A,B11,B12,B21,B22,C11,C12,C21,C22);

%Equ of type Ac' + Bc = 0 can be block partitioned like 
% (A|Abd)*(c;cbd)' + (B|Bbd)*(c;cbd) = 0 which after rewriting becomes
% Ac'+Bc = -Abd*cbd' - Bbd*cbd.  You can use the Nodal_Bud list division to
% tell you where the bd vertices begin and end.

%We no longer need these and so we clear the variables so I can write with
%A,B as in previous comment.
clear A B11 B12 B21 B22 C11 C12 C21 C22

A = P1(:,1:Nodal_Bud(2,1));
Abd = P1(:,Nodal_Bud(2,1)+1:Nodal_Bud(4,1));
B = P2(:,1:Nodal_Bud(2,1));
Bbd = P2(:,Nodal_Bud(2,1)+1:Nodal_Bud(4,1));
Ainv = A^(-1);
%Recall our boundary values are constant in time, so our equation becomes
%Ac' + Bc = -Abd*0 - Bbd*cbd = 0 - Bbd*cbd
%         = -Bbd*cbd
%where the above coefficients are constant over time.

%setup time axis
n = floor(tfinal/tstep);
time = 0:tstep:tstep*n;
if floor(tfinal/tstep) ~= tfinal/tstep
time = [time tfinal]; %Augment for the uneven final increment.
end

%We run the integration with Matlab's built in ODE45.  If you want to
%change it's options, you can do it by writing options = odeset(...) and
%plugging this output into the options component of ode45.

                                                      
options = odeset('RelTol',2,'AbsTol',.1);
odefun = @(t,y) Ainv*(-B*y - Bbd*cbd); %c' = A^-1(-Bc - Bbd*cbd)

[tout, yout] = ode45(odefun, time, cinitial);
yout = yout'; %Have nodal values be listed column-wise.
bottom = zeros(Nodal_Bud(4,1)-Nodal_Bud(2,1),size(tout,1)); %Append the cbd values to bottom of yout.
for i=1:size(tout,1)
    bottom(:,i) = cbd;
end

yout = [yout; bottom]; %cbd values appended.

end

